<?php

/**
 * @copyright
 * @package    Easy Joomla Backup Pro - EJB for Joomla! 5
 * @author     Viktor Vogel <admin@kubik-rubik.de>
 * @version    5.1.0.0-PRO - 2024-07-28
 * @link       https://kubik-rubik.de/ejb-easy-joomla-backup
 *
 * @license    GNU/GPL
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

namespace KubikRubik\Component\EasyJoomlaBackup\Administrator\Model;

defined('_JEXEC') || die('Restricted access');

use Exception;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\CMS\{Application\CMSApplication, Date\Date, Factory, Uri\Uri, Language\Text, User\UserHelper, Component\ComponentHelper};
use Joomla\Database\DatabaseDriver;
use Joomla\Filesystem\{File, Folder};
use Joomla\Input\Input;
use Joomla\Registry\Registry;
use KubikRubik\Component\EasyJoomlaBackup\Administrator\Helper\DropboxHelper;
use KubikRubik\Component\EasyJoomlaBackup\Administrator\Helper\EasyJoomlaBackupHelper;
use KubikRubik\Component\EasyJoomlaBackup\Administrator\Table\CreateTable;
use phpseclib3\Net\SFTP;
use RuntimeException;
use ZipArchive;

use function count;
use function defined;
use function in_array;

/**
 * Class CreateModel
 *
 * @version 5.0.1.0-PRO
 * @since   5.0.0.0-PRO
 */
class CreateModel extends BaseDatabaseModel
{
    /**
     * @var string DEFAULT_PREFIX
     * @since 5.0.0.0-PRO
     */
    public const DEFAULT_PREFIX = 'easy-joomla-backup-pro';

    /**
     * @var CMSApplication $app
     * @since 5.0.0.0-PRO
     */
    private $app;

    /**
     * @var DatabaseDriver $db
     * @since 5.0.0.0-PRO
     */
    private $db;

    /**
     * @var Input $input
     * @since 5.0.0.0-PRO
     */
    private Input $input;

    /**
     * @var Date $backupDatetime
     * @since 5.0.0.0-PRO
     */
    private Date $backupDatetime;

    /**
     * @var Registry $params
     * @since 5.0.0.0-PRO
     */
    private Registry $params;

    /**
     * @var string $backupFolder
     * @since 5.0.0.0-PRO
     */
    private string $backupFolder;

    /**
     * @var string $backupPath
     * @since 5.0.0.0-PRO
     */
    private string $backupPath;

    /**
     * @var bool $encryptArchive
     * @since 5.0.0.0-PRO
     */
    private bool $encryptArchive = false;

    /**
     * @var int $encryptionMethod
     * @since 5.0.0.0-PRO
     */
    private int $encryptionMethod;

    /**
     * @var string $encryptionPassword
     * @since 5.0.0.0-PRO
     */
    private string $encryptionPassword = '';

    /**
     * @var bool $externalAttributes
     * @since 5.0.0.0-PRO
     */
    private bool $externalAttributes = false;

    /**
     * @var int $maximumExecutionLevel
     * @since 5.0.0.0-PRO
     */
    private int $maximumExecutionLevel;

    /**
     * @var float $maximumExecutionTime
     * @since 5.0.0.0-PRO
     */
    private float $maximumExecutionTime;

    /**
     * @var int $maximumExecutionTimeDefault
     * @since 5.0.0.0-PRO
     */
    private int $maximumExecutionTimeDefault = 30;

    /**
     * @var int $maximumListLimit
     * @since 5.0.0.0-PRO
     */
    private int $maximumListLimit = 100000;

    /**
     * @var int $maximumInsertLimit
     * @since 5.0.0.0-PRO
     */
    private int $maximumInsertLimit = 250;

    /**
     * @var string $fileName
     * @since 5.0.0.0-PRO
     */
    private string $fileName = '';

    /**
     * @var string $fileNameDatabase
     * @since 5.0.0.0-PRO
     */
    private string $fileNameDatabase = '';

    /**
     * @var object $iterator
     * @since 5.0.0.0-PRO
     */
    private object $iterator;

    /**
     * @var array $dataStored
     * @since 5.0.0.0-PRO
     */
    private array $dataStored = [];

    /**
     * @var string $executionSource
     * @since 5.0.0.0-PRO
     */
    private string $executionSource = '';

    /**
     * @var array $postProcessData
     * @since 5.0.0.0-PRO
     */
    private array $postProcessData = [];

    /**
     * @var int $maximumFilesPerIteration
     * @since 5.0.0.0-PRO
     */
    private int $maximumFilesPerIteration = 500;

    /**
     * @var int $maximumFilesPerIterationCounter
     * @since 5.0.0.0-PRO
     */
    private int $maximumFilesPerIterationCounter = 0;

    /**
     * @var array $excludeFiles
     * @since 5.0.0.0-PRO
     */
    private array $excludeFiles = [];

    /**
     * @var array $excludeFolders
     * @since 5.0.0.0-PRO
     */
    private array $excludeFolders = [];

    /**
     * @var array $excludeFoldersCreateEmpty
     * @since 5.0.0.0-PRO
     */
    private array $excludeFoldersCreateEmpty = [];

    /**
     * @var bool $excludeFilesWildcard
     * @since 5.0.0.0-PRO
     */
    private bool $excludeFilesWildcard = false;

    /**
     * @var array $addDbTables
     * @since 5.0.0.0-PRO
     */
    private array $addDbTables = [];

    /**
     * @var string $dbPrefix
     * @since 5.0.0.0-PRO
     */
    private string $dbPrefix = '';

    /**
     * @var bool $allDbTables
     * @since 5.0.0.0-PRO
     */
    private bool $allDbTables = false;

    /**
     * @var bool $addDropStatement
     * @since 5.0.0.0-PRO
     */
    private bool $addDropStatement = true;

    /**
     * @var string $pathRoot
     * @since 5.0.1.0-PRO
     */
    private string $pathRoot;

    /**
     * CreateModel constructor
     *
     * @throws Exception
     * @version 5.0.1.0-PRO
     * @since   5.0.0.0-PRO
     */
    public function __construct()
    {
        parent::__construct();

        $this->app = Factory::getApplication();
        $this->input = $this->app->getInput();
        $this->params = ComponentHelper::getParams('com_easyjoomlabackup');
        $this->backupFolder = EasyJoomlaBackupHelper::getBackupStorageLocation(true);
        $this->backupDatetime = Factory::getDate('now', $this->app->get('offset'));
        $this->maximumExecutionTime = (float)$this->getMaximumExecutionTime();
        $this->maximumExecutionLevel = (int)$this->params->get('maximumExecutionLevel', 8);
        $this->maximumFilesPerIteration = (int)$this->params->get('maximumFilesPerIteration', 500);
        $this->pathRoot = str_replace('\\', '/', JPATH_ROOT);
        $this->prepareDatabase();
    }

    /**
     * Gets the maximum execution time for each batch process
     *
     * @return float|int
     * @since 5.0.0.0-PRO
     */
    private function getMaximumExecutionTime(): float|int
    {
        $maximumExecutionTime = (int)$this->params->get('maximumExecutionTime', 0);

        if ($maximumExecutionTime !== 0) {
            return $maximumExecutionTime;
        }

        $maximumExecutionTimeServer = (int)ini_get('max_execution_time');

        if (!empty($maximumExecutionTimeServer)) {
            return $maximumExecutionTimeServer / 3;
        }

        return $this->maximumExecutionTimeDefault;
    }

    /**
     * Prepares the database for the backup process
     *
     * @return void
     * @since 5.0.0.0-PRO
     */
    private function prepareDatabase(): void
    {
        $this->db = Factory::getContainer()->get('DatabaseDriver');
        $this->db->setUtf();
        $this->dbPrefix = $this->db->getPrefix();
        $this->addDbTables = $this->getAddDbTables();
        $this->allDbTables = (bool)$this->params->get('addDbTablesAll', false);
        $this->addDropStatement = (bool)$this->params->get('addDropStatement', true);
    }

    /**
     * Gets the list of additional database tables to be included in the backup
     *
     * @return array
     * @since 5.0.0.0-PRO
     */
    private function getAddDbTables(): array
    {
        $addDbTables = $this->params->get('addDbTables');

        if ($addDbTables === null) {
            return [];
        }

        return array_filter(array_map('trim', explode("\n", $addDbTables)));
    }

    /**
     * Main function for the backup process
     *
     * @param string $type
     * @param string $hash
     * @param string $password
     *
     * @return array
     * @throws Exception
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    public function createBackupAjax(string $type, string $hash, string $password): array
    {
        if (!class_exists('ZipArchive')) {
            return [];
        }

        $this->executionSource = $type;
        $this->externalAttributes = $this->checkExternalAttributes();
        $this->encryptArchive = EasyJoomlaBackupHelper::checkEncryptArchive();
        $this->encryptionMethod = $this->setEncryptionMethod();
        $this->encryptionPassword = $this->setEncryptionPassword($password);

        $status = true;
        $statusDb = true;

        $this->fileName = $this->createFilenameAjax($hash, (bool)$password);
        $this->backupPath = $this->backupFolder . $this->fileName;

        if (!is_file($this->backupPath)) {
            $this->prepareBackupProcess($type, $hash);

            $message = ($type === 'databasebackup') ? Text::_('COM_EASYJOOMLABACKUP_BACKUPMODAL_INITIALISE_DB') : Text::sprintf('COM_EASYJOOMLABACKUP_BACKUPMODAL_LASTFOLDER', 'root');

            return [
                'hash' => $hash,
                'finished' => false,
                'percent' => 0,
                'message' => $message,
            ];
        }

        if ($type === 'filebackup' || $type === 'fullbackup') {
            $fileBackupDone = (bool)$this->app->getUserState('ejb.' . $hash . '.fileBackupDone', false);

            if (!$fileBackupDone) {
                $status = $this->createBackupZipArchiveFilesAjax($hash);

                $foldersIteration = (array)$this->app->getUserState('ejb.' . $hash . '.foldersIteration', []);
                $foldersIterationCount = (int)$this->app->getUserState('ejb.' . $hash . '.foldersIterationCount', 1);
                $message = Text::sprintf('COM_EASYJOOMLABACKUP_BACKUPMODAL_LASTFOLDER', $foldersIteration[0]['relname'] ?? '');
                $totalPercentage = ($type === 'fullbackup') ? 90 : 100;

                if ($status) {
                    $this->app->setUserState('ejb.' . $hash . '.fileBackupDone', true);
                    $foldersIteration = [];
                    $foldersIterationCount = 1;
                    $message = ($type === 'fullbackup') ? Text::_('COM_EASYJOOMLABACKUP_BACKUPMODAL_FILEBACKUPDONE_DB') : Text::_('COM_EASYJOOMLABACKUP_BACKUPMODAL_FILEBACKUPDONE');
                }

                return [
                    'hash' => $hash,
                    'finished' => false,
                    'percent' => $totalPercentage - round(count($foldersIteration) * $totalPercentage / $foldersIterationCount),
                    'message' => $message,
                ];
            }
        }

        if ($type === 'databasebackup' || $type === 'fullbackup') {
            $this->setFileNameDatabase();
            $dbBackupDone = (bool)$this->app->getUserState('ejb.' . $hash . '.dbBackupDone', false);

            if (!$dbBackupDone) {
                $statusDb = $this->createBackupSqlArchiveDatabaseAjax($hash);
                $dbTables = (array)$this->app->getUserState('ejb.' . $hash . '.dbTables', []);
                $dbTablesCount = (int)$this->app->getUserState('ejb.' . $hash . '.dbTablesCount', 1);
                $message = Text::sprintf('COM_EASYJOOMLABACKUP_BACKUPMODAL_LASTTABLE', current($dbTables));
                $totalPercentage = ($type === 'fullbackup') ? 10 : 100;

                if ($statusDb) {
                    $this->app->setUserState('ejb.' . $hash . '.dbBackupDone', true);
                    $dbTables = [];
                    $dbTablesCount = 1;
                    $message = ($type === 'fullbackup') ? Text::_('COM_EASYJOOMLABACKUP_BACKUPMODAL_FILEDATABASEBACKUPDONE_DB') : Text::_('COM_EASYJOOMLABACKUP_BACKUPMODAL_DATABASEBACKUPDONE');
                }

                return [
                    'hash' => $hash,
                    'finished' => false,
                    'percent' => ($type === 'fullbackup' ? 90 : 0) + $totalPercentage - round(count($dbTables) * $totalPercentage / $dbTablesCount),
                    'message' => $message,
                ];
            }

            $this->createBackupSqlArchiveDatabasePostProcess($hash);
            $this->createBackupZipArchiveDatabaseAjax();
        }

        if (!empty($status) && !empty($statusDb)) {
            $table = new CreateTable($this->db);

            $data = [];
            $data['date'] = $this->backupDatetime->toSql();
            $data['type'] = $type;
            $data['name'] = $this->fileName;
            $data['size'] = filesize($this->backupFolder . $this->fileName);

            $startProcess = $this->app->getUserState('ejb.' . $hash . '.startProcess', microtime(true));
            $data['duration'] = round(microtime(true) - $startProcess, 2);
            $data['comment'] = $this->input->get('comment', '', 'STRING');

            if (!$table->save($data)) {
                throw new RuntimeException(Text::_('JERROR_AN_ERROR_HAS_OCCURRED'), 404);
            }

            return [
                'hash' => $hash,
                'finished' => true,
                'percent' => 100,
                'backupLastFile' => Text::_('COM_EASYJOOMLABACKUP_BACKUPMODAL_DONE'),
            ];
        }

        return [];
    }

    /**
     * Check whether required function to set the permission rights on UNIX systems is available
     * Since PHP 5 >= 5.6.0, PHP 7, PECL zip >= 1.12.4
     *
     * @return bool
     * @since 5.0.0.0-PRO
     */
    private function checkExternalAttributes(): bool
    {
        $zipObject = new ZipArchive();

        return method_exists($zipObject, 'setExternalAttributesName');
    }

    /**
     * Sets the encryption method for password protection process
     *
     * @return int
     * @since 5.0.0.0-PRO
     */
    private function setEncryptionMethod(): int
    {
        if (!$this->encryptArchive) {
            return 0;
        }

        $encryptionMethod = $this->params->get('encryptionMethodAlgorithm', 'EM_AES_128');

        if ($encryptionMethod === 'EM_AES_256') {
            return ZipArchive::EM_AES_256;
        }

        if ($encryptionMethod === 'EM_AES_192') {
            return ZipArchive::EM_AES_192;
        }

        return ZipArchive::EM_AES_128;
    }

    /**
     * Sets the encryption password for the backup process
     *
     * @param string $password
     *
     * @return string
     * @since 5.0.0.0-PRO
     */
    private function setEncryptionPassword(string $password): string
    {
        return $password;
    }

    /**
     * Creates a filename for the backup archive from the URL, the date and a random string
     *
     * @param string $hash
     * @param bool   $password
     *
     * @return string
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    private function createFilenameAjax(string &$hash, bool $password = false): string
    {
        if (!empty($hash)) {
            $fileName = $this->app->getUserState('ejb.' . $hash . '.filename');

            if (!empty($fileName)) {
                return $fileName;
            }
        }

        $fileName = $this->getFileNamePrefix() . '_' . $this->backupDatetime->format('Y-m-d_H-i-s', true);

        if (is_file($this->backupFolder . $fileName)) {
            $fileName = $this->createFilenameAjax($hash);
        }

        $addSuffix = $this->params->get('addSuffixArchive', 1);

        if (!empty($addSuffix)) {
            $fileName .= '_' . UserHelper::genRandomPassword(16);
        }

        if ($password) {
            $fileName .= '.secure';
        }

        $fileName .= '.zip';

        $hash = md5($fileName);
        $this->app->setUserState('ejb.' . $hash . '.filename', $fileName);

        return $fileName;
    }

    /**
     * Gets the filename prefix for the archive files
     *
     * @param bool $hostOnly
     *
     * @return string
     * @since 5.0.0.0-PRO
     */
    private function getFileNamePrefix(bool $hostOnly = false): string
    {
        if ($this->params->get('prefixArchive')) {
            return strtolower(preg_replace('@\s+@', '-', $this->params->get('prefixArchive')));
        }

        if ($hostOnly) {
            return (string)Uri::getInstance()->getHost();
        }

        $root = (string)Uri::root();

        if (!empty($root)) {
            return implode('-', array_filter(explode('/', str_replace(['http://', 'https://', ':'], '', $root))));
        }

        return self::DEFAULT_PREFIX;
    }

    /**
     * Prepares the backup process with all required states
     *
     * @param string $backupType
     * @param string $hash
     *
     * @return void
     * @version 5.0.1.0-PRO
     * @since   5.0.0.0-PRO
     */
    private function prepareBackupProcess(string $backupType, string $hash): void
    {
        $this->app->setUserState('ejb.' . $hash . '.startProcess', microtime(true));

        if ($backupType === EasyJoomlaBackupHelper::BACKUP_TYPE_DATABASE || $backupType === EasyJoomlaBackupHelper::BACKUP_TYPE_FULL) {
            $tables = $this->getDatabaseTables();

            $this->app->setUserState('ejb.' . $hash . '.dbTables', $tables);
            $this->app->setUserState('ejb.' . $hash . '.dbTablesCount', count($tables));
            $this->app->setUserState('ejb.' . $hash . '.dbBackupDone', false);
        }

        if ($backupType === EasyJoomlaBackupHelper::BACKUP_TYPE_FILE || $backupType === EasyJoomlaBackupHelper::BACKUP_TYPE_FULL) {
            $filesRoot = Folder::files($this->pathRoot, '.', false, false, [], []);
            $foldersIteration = Folder::listFolderTree($this->pathRoot, '.', $this->maximumExecutionLevel);

            foreach ($foldersIteration as &$folderIteration) {
                $folderIteration['relname'] = str_replace('\\', '/', $folderIteration['relname']);

                if (strncmp($folderIteration['relname'], '/', 1) === 0) {
                    $folderIteration['relname'] = substr($folderIteration['relname'], 1);
                }
            }

            unset($folderIteration);

            $this->app->setUserState('ejb.' . $hash . '.filesRoot', $filesRoot);
            $this->app->setUserState('ejb.' . $hash . '.foldersIteration', $foldersIteration);
            $this->app->setUserState('ejb.' . $hash . '.foldersIterationCount', count($foldersIteration));
            $this->app->setUserState('ejb.' . $hash . '.fileBackupDone', false);
        }

        File::write($this->backupPath, '');
    }

    /**
     * Gets the database tables to be included in the backup
     *
     * @return array
     * @since 5.0.0.0-PRO
     */
    private function getDatabaseTables(): array
    {
        $tables = $this->db->getTableList();

        if ($this->allDbTables) {
            return $tables;
        }

        $prefix = $this->dbPrefix;
        $additionalDbTables = $this->addDbTables;

        return array_filter(
            $tables,
            static function ($table) use ($prefix, $additionalDbTables) {
                return str_starts_with($table, $prefix) || in_array($table, $additionalDbTables, true);
            }
        );
    }

    /**
     * Creates the archive file of all files from the Joomla! installation with a possible exclusion of files and folders
     *
     * @param string $hash
     *
     * @return bool
     * @throws Exception
     * @version 5.0.1.0-PRO
     * @since   5.0.0.0-PRO
     */
    private function createBackupZipArchiveFilesAjax(string $hash): bool
    {
        $this->excludeFiles = $this->getExcludeFiles();
        $this->excludeFolders = $this->getExcludeFolders();
        $this->excludeFoldersCreateEmpty = $this->getExcludeFoldersEmpty();

        if (!@opendir($this->pathRoot)) {
            throw new RuntimeException('Error: Root path could not be opened!');
        }

        $filesRoot = (array)$this->app->getUserState('ejb.' . $hash . '.filesRoot', []);

        if (!empty($filesRoot)) {
            $zipFile = EasyJoomlaBackupHelper::openZipArchive($this->backupPath);

            foreach ($filesRoot as $file) {
                if ($this->isFileExcluded($file)) {
                    continue;
                }

                $zipFile->addFile($this->pathRoot . '/' . $file, $file);
                $this->setExternalAttributes($zipFile, $file, fileperms($this->pathRoot . '/' . $file));
                $this->encryptArchive($zipFile, $file);
            }

            EasyJoomlaBackupHelper::closeZipArchive($zipFile);

            if ($zipFile->status !== 0) {
                throw new RuntimeException('Error: Zip status is not correct!');
            }

            $this->app->setUserState('ejb.' . $hash . '.filesRoot', []);

            return false;
        }

        $foldersIteration = (array)$this->app->getUserState('ejb.' . $hash . '.foldersIteration', []);

        if (!empty($foldersIteration)) {
            $zipFolder = EasyJoomlaBackupHelper::openZipArchive($this->backupPath);

            while ($this->maximumExecutionTime > 0 && !empty($foldersIteration)) {
                $startTime = microtime(true);
                $folderIteration = array_shift($foldersIteration);
                $recursive = false;

                if (count(explode('/', $folderIteration['relname'])) === $this->maximumExecutionLevel) {
                    $recursive = true;
                }

                if (!$this->zipFoldersAndFilesRecursiveAjax($zipFolder, $this->pathRoot . '/' . $folderIteration['relname'], $folderIteration['relname'], $recursive)) {
                    $this->maximumFilesPerIterationCounter = 0;
                    EasyJoomlaBackupHelper::closeZipArchive($zipFolder);
                    $zipFolder = EasyJoomlaBackupHelper::openZipArchive($this->backupPath);
                    array_unshift($foldersIteration, $folderIteration);
                }

                $endTime = microtime(true);
                $this->maximumExecutionTime -= $this->ceilDecimalDigits($endTime - $startTime);
            }

            EasyJoomlaBackupHelper::closeZipArchive($zipFolder);

            if ($zipFolder->status !== 0) {
                throw new RuntimeException('Error: Zip status is not correct!');
            }

            if (count($foldersIteration) >= 1) {
                $this->app->setUserState('ejb.' . $hash . '.foldersIteration', $foldersIteration);

                return false;
            }
        }

        return true;
    }

    /**
     * Gets a list of files that should be excluded in the backup archive
     *
     * @return array
     * @since 5.0.0.0-PRO
     */
    private function getExcludeFiles(): array
    {
        $excludeFiles = $this->params->get('excludeFiles');

        if ($excludeFiles === null) {
            return [];
        }

        $excludeFiles = array_filter(array_map('trim', explode("\n", $excludeFiles)));

        foreach ($excludeFiles as $excludeFile) {
            if (str_contains($excludeFile, '*')) {
                $this->excludeFilesWildcard = true;

                break;
            }
        }

        return $excludeFiles;
    }

    /**
     * Gets a list of folder that should be excluded in the backup archive
     *
     * @return array
     * @since 5.0.0.0-PRO
     */
    private function getExcludeFolders(): array
    {
        $excludeFolders = $this->params->get('excludeFolders');

        if ($excludeFolders === null) {
            return [];
        }

        return array_filter(array_map('trim', explode("\n", $excludeFolders)));
    }

    /**
     * Gets a list of folder that should be empty in the backup archive
     *
     * @return array
     * @since 5.0.0.0-PRO
     */
    private function getExcludeFoldersEmpty(): array
    {
        return [
            'administrator/components/com_easyjoomlabackup/backups',
            'cache',
            'tmp',
            'administrator/cache',
            EasyJoomlaBackupHelper::getBackupStorageLocation(false, true),
        ];
    }

    /**
     * Is the file excluded from the backup archive
     *
     * @param string $file
     *
     * @return bool
     * @since 5.0.0.0-PRO
     */
    private function isFileExcluded(string $file): bool
    {
        if (empty($this->excludeFiles)) {
            return false;
        }

        if ($this->excludeFiles === []) {
            return false;
        }

        if (!$this->excludeFilesWildcard) {
            if (in_array($file, $this->excludeFiles, true)) {
                return true;
            }

            return false;
        }

        foreach ($this->excludeFiles as $excludeFile) {
            if (str_contains($excludeFile, '*')) {
                if (fnmatch($excludeFile, $file)) {
                    return true;
                }

                continue;
            }

            if ($excludeFile === $file) {
                return true;
            }
        }

        return false;
    }

    /**
     * Sets external attributes
     *
     * @param object $zipObject
     * @param string $fileName
     * @param int    $filePermission
     *
     * @since 5.0.0.0-PRO
     */
    private function setExternalAttributes(object $zipObject, string $fileName, int $filePermission): void
    {
        if ($this->externalAttributes) {
            $zipObject->setExternalAttributesName($fileName, ZipArchive::OPSYS_UNIX, $filePermission << 16);
        }
    }

    /**
     * Encrypts added file in the archive
     *
     * @param object $zipObject
     * @param string $fileName
     *
     * @since 5.0.0.0-PRO
     */
    private function encryptArchive(object $zipObject, string $fileName): void
    {
        if ($this->encryptArchive && !empty($this->encryptionPassword)) {
            $zipObject->setEncryptionName($fileName, $this->encryptionMethod, $this->encryptionPassword);
        }
    }

    /**
     * Loads all files and (sub-)folders for the zip archive recursively
     *
     * @param object $zip
     * @param string $folder
     * @param string $folderRelative
     * @param bool   $recursive
     *
     * @return bool
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    private function zipFoldersAndFilesRecursiveAjax(object $zip, string $folder, string $folderRelative, bool $recursive): bool
    {
        if (!empty($this->excludeFolders)) {
            foreach ($this->excludeFolders as $excludeFolder) {
                if (stripos($folderRelative, $excludeFolder) === 0) {
                    return true;
                }
            }
        }

        if (in_array($folderRelative, $this->excludeFoldersCreateEmpty, true)) {
            $zip->addEmptyDir($folderRelative . '/');
            $this->setExternalAttributes($zip, $folderRelative . '/', 16877);

            $zip->addFromString($folderRelative . '/index.html', '');
            $this->setExternalAttributes($zip, $folderRelative . '/index.html', 33188);
            $this->encryptArchive($zip, $folderRelative . '/index.html');

            if ($folderRelative === 'administrator/components/com_easyjoomlabackup/backups' || $folderRelative === EasyJoomlaBackupHelper::getBackupStorageLocation(false, true)) {
                $zip->addFromString($folderRelative . '/.htaccess', 'Deny from all');
                $this->setExternalAttributes($zip, $folderRelative . '/.htaccess', 33060);
                $this->encryptArchive($zip, $folderRelative . '/.htaccess');
            }

            return true;
        }

        foreach ($this->excludeFoldersCreateEmpty as $excludeFolderCreateEmpty) {
            if (stripos($folderRelative, $excludeFolderCreateEmpty) === 0) {
                return true;
            }
        }

        $zip->addEmptyDir($folderRelative . '/');
        $this->setExternalAttributes($zip, $folderRelative . '/', fileperms($folder . '/'));

        if (!$dir = @opendir($folder)) {
            return false;
        }

        while ($file = readdir($dir)) {
            if ($file !== '.' && $file !== '..' && is_dir($folder . '/' . $file)) {
                if (in_array($folderRelative . '/' . $file, $this->excludeFoldersCreateEmpty, true)) {
                    $zip->addEmptyDir($folderRelative . '/' . $file . '/');
                    $this->setExternalAttributes($zip, $folderRelative . '/' . $file . '/', 16877);

                    $zip->addFromString($folderRelative . '/' . $file . '/index.html', '');
                    $this->setExternalAttributes($zip, $folderRelative . '/' . $file . '/index.html', 33188);
                    $this->encryptArchive($zip, $folderRelative . '/' . $file . '/index.html');

                    if ($folderRelative . '/' . $file === 'administrator/components/com_easyjoomlabackup/backups' || $folderRelative . '/' . $file === EasyJoomlaBackupHelper::getBackupStorageLocation(false, true)) {
                        $zip->addFromString($folderRelative . '/' . $file . '/.htaccess', 'Deny from all');
                        $this->setExternalAttributes($zip, $folderRelative . '/' . $file . '/.htaccess', 33060);
                        $this->encryptArchive($zip, $folderRelative . '/' . $file . '/.htaccess');
                    }

                    continue;
                }

                if (!empty($this->excludeFolders) && in_array($folderRelative . '/' . $file, $this->excludeFolders, true)) {
                    continue;
                }

                $zip->addEmptyDir($folderRelative . '/' . $file . '/');
                $this->setExternalAttributes($zip, $folderRelative . '/' . $file . '/', fileperms($folder . '/' . $file . '/'));

                if ($recursive) {
                    $this->zipFoldersAndFilesRecursiveAjax($zip, $folder . '/' . $file, $folderRelative . '/' . $file, true);

                    if ($this->maximumFilesPerIterationCounter >= $this->maximumFilesPerIteration) {
                        return false;
                    }
                }
            } elseif (is_file($folder . '/' . $file)) {
                if ($this->isFileExcluded($folderRelative . '/' . $file)) {
                    continue;
                }

                if ($zip->locateName($folderRelative . '/' . $file) !== false) {
                    continue;
                }

                if ($this->maximumFilesPerIterationCounter >= $this->maximumFilesPerIteration) {
                    return false;
                }

                $zip->addFile($folder . '/' . $file, $folderRelative . '/' . $file);
                $this->setExternalAttributes($zip, $folderRelative . '/' . $file, fileperms($folder . '/' . $file));
                $this->encryptArchive($zip, $folderRelative . '/' . $file);

                $this->maximumFilesPerIterationCounter++;
            }
        }

        closedir($dir);

        return true;
    }

    /**
     * Helper function to apply the ceil function to float numbers with decimal digits (default precision = 2)
     *
     * @param float $value
     * @param int   $precision
     *
     * @return float|int
     * @since 5.0.0.0-PRO
     */
    private function ceilDecimalDigits(float $value, int $precision = 2): float|int
    {
        return ceil($value * (10 ** $precision)) / (10 ** $precision);
    }

    /**
     * Sets the database filename
     *
     * @since 5.0.0.0-PRO
     */
    private function setFileNameDatabase(): void
    {
        $this->fileNameDatabase = str_replace('.zip', '', $this->fileName) . '.sql';
    }

    /**
     * Creates a complete dump of the Joomla! database as a SQL file
     *
     * @param string $hash
     *
     * @return bool
     * @version 5.0.1.0-PRO
     * @since   5.0.0.0-PRO
     */
    private function createBackupSqlArchiveDatabaseAjax(string $hash): bool
    {
        if (!is_file($this->backupFolder . $this->fileNameDatabase)) {
            $data = EasyJoomlaBackupHelper::getDatabaseDumpHeader();
            $this->writeDatabaseFile($data);
        }

        $dbTables = (array)$this->app->getUserState('ejb.' . $hash . '.dbTables', []);

        while ($this->maximumExecutionTime > 0 && !empty($dbTables)) {
            $startTime = microtime(true);
            $dbTable = array_shift($dbTables);

            $this->backupDatabaseAjax($dbTable, $hash);

            $endTime = microtime(true);
            $this->maximumExecutionTime -= $this->ceilDecimalDigits($endTime - $startTime);
        }

        if (count($dbTables) >= 1) {
            $this->app->setUserState('ejb.' . $hash . '.dbTables', $dbTables);

            return false;
        }

        return true;
    }

    /**
     * Adds the data to the temporary dump file and cleans the data string
     *
     * @param string $data
     *
     * @since 5.0.0.0-PRO
     */
    private function writeDatabaseFile(string &$data): void
    {
        file_put_contents($this->backupFolder . $this->fileNameDatabase, $data, FILE_APPEND);
        $data = '';
    }

    /**
     * Creates an SQL Dump of the Joomla! database and add it directly to the archive
     *
     * @param string $dbTable
     * @param string $hash
     *
     * @return void
     * @since 5.0.0.0-PRO
     */
    private function backupDatabaseAjax(string $dbTable, string $hash): void
    {
        if (!$this->isDbTableValid($dbTable)) {
            return;
        }

        if ($this->addDropStatement) {
            $data = 'DROP TABLE IF EXISTS ' . $this->db->quoteName($dbTable) . ';' . "\n";
            $this->writeDatabaseFile($data);
        }

        $this->db->setQuery('SHOW CREATE TABLE ' . $this->db->quoteName($dbTable));
        $rowCreate = $this->db->loadRow();
        $this->checkPostProcessDataCreateTable($rowCreate, $hash);

        $data = $rowCreate[1] . ";\n\n";
        $this->writeDatabaseFile($data);

        $tableInformation = $this->db->getTableColumns($dbTable);
        $numFields = count($tableInformation);
        $tableColumnTypes = array_values($tableInformation);

        $this->db->setQuery('SELECT * FROM ' . $this->db->quoteName($dbTable));
        $this->db->execute();
        $count = $this->db->getNumRows();

        if ($count > 0) {
            $passes = (int)ceil($count / $this->maximumListLimit);
            $this->iterator = $this->db->getIterator();
            $this->db->setQuery('SELECT * FROM ' . $this->db->quoteName($dbTable));

            for ($round = 0; $round < $passes; $round++) {
                $rowList = $this->getRowList($passes);
                $this->addRowEntries($rowList, $numFields, $tableColumnTypes, $dbTable);
            }
        }

        $data .= "\n\n";
        $this->writeDatabaseFile($data);
    }

    /**
     * Is the database table valid for the backup?
     *
     * @param string $dbTable
     *
     * @return bool
     * @since 5.0.0.0-PRO
     */
    private function isDbTableValid(string $dbTable): bool
    {
        if (stripos($dbTable, $this->dbPrefix) !== false) {
            return true;
        }

        if ($this->allDbTables) {
            return true;
        }

        if (in_array($dbTable, $this->addDbTables, true)) {
            return true;
        }

        return false;
    }

    /**
     * Checks the create database command for post process rules
     *
     * @param array  $rowCreate
     * @param string $hash
     *
     * @since 5.0.0.0-PRO
     */
    private function checkPostProcessDataCreateTable(array &$rowCreate, string $hash): void
    {
        if (empty($rowCreate) || count($rowCreate) !== 2) {
            return;
        }

        if (!str_contains($rowCreate[1], 'CONSTRAINT')) {
            return;
        }

        [$tableName, $createTableCommand] = $rowCreate;
        $postProcessData = $this->getPostProcessData($hash);

        preg_match('@\(\n?(.*)\n?\)@s', $createTableCommand, $matchColumnsDefinition);
        $matchColumnsDefinitionArray = array_filter(array_map('trim', explode("\n", $matchColumnsDefinition[1])));

        foreach ($matchColumnsDefinitionArray as $matchColumnsDefinitionArrayKey => &$matchColumnsDefinitionArrayValue) {
            if (strncmp($matchColumnsDefinitionArrayValue, 'CONSTRAINT', 10) === 0) {
                $postProcessData[] = [
                    'type' => 'constraint',
                    'table' => $tableName,
                    'command' => EasyJoomlaBackupHelper::removeTrailingComma($matchColumnsDefinitionArrayValue),
                ];

                unset($matchColumnsDefinitionArray[$matchColumnsDefinitionArrayKey]);
                continue;
            }

            $matchColumnsDefinitionArrayValue = '  ' . EasyJoomlaBackupHelper::removeTrailingComma($matchColumnsDefinitionArrayValue);
        }

        unset($matchColumnsDefinitionArrayValue);

        $this->setPostProcessData($postProcessData, $hash);
        $rowCreate[1] = str_replace($matchColumnsDefinition[1], implode(',' . "\n", $matchColumnsDefinitionArray) . "\n", $createTableCommand);
    }

    /**
     * Sets the post process data depending on the execution source
     *
     * @param string $hash
     *
     * @return array
     * @since 5.0.0.0-PRO
     */
    private function getPostProcessData(string $hash): array
    {
        if ($this->executionSource === 'cli') {
            return $this->postProcessData;
        }

        return (array)$this->app->getUserState('ejb.' . $hash . '.postProcessData', []);
    }

    /**
     * Sets the post process data depending on the execution source
     *
     * @param array  $postProcessData
     * @param string $hash
     *
     * @since 5.0.0.0-PRO
     */
    private function setPostProcessData(array $postProcessData, string $hash): void
    {
        if ($this->executionSource === 'cli') {
            $this->postProcessData = $postProcessData;

            return;
        }

        $this->app->setUserState('ejb.' . $hash . '.postProcessData', $postProcessData);
    }

    /**
     * Gets a list of database entries from the table query - using an iterator if too many entries
     *
     * @param int $passes
     *
     * @return array
     * @version 5.0.1.0-PRO
     * @since   5.0.0.0-PRO
     */
    private function getRowList(int $passes): array
    {
        if ($passes === 1) {
            return $this->db->loadRowList();
        }

        $rowList = [];
        $iteratorCount = 0;

        foreach ($this->iterator as $row) {
            if ($iteratorCount >= $this->maximumListLimit) {
                break;
            }

            if (!empty($row)) {
                $rowList[] = array_values((array)$row);
                $iteratorCount++;
            }
        }

        return $rowList;
    }

    /**
     * Adds insert values to the dump file
     *
     * @param array  $rowList
     * @param int    $numFields
     * @param array  $tableColumns
     * @param string $table
     *
     * @since 5.0.0.0-PRO
     */
    private function addRowEntries(array $rowList, int $numFields, array $tableColumns, string $table): void
    {
        $countEntries = 0;
        $count = count($rowList);
        $data = 'INSERT INTO ' . $this->db->quoteName($table) . ' VALUES' . "\n";

        foreach ($rowList as $row) {
            $countEntries++;
            $data .= '(';

            for ($j = 0; $j < $numFields; $j++) {
                if ($row[$j] === null) {
                    $data .= 'NULL';

                    if ($j < ($numFields - 1)) {
                        $data .= ', ';
                    }

                    continue;
                }

                if (empty($row[$j])) {
                    $data .= '\'\'';

                    if ($j < ($numFields - 1)) {
                        $data .= ', ';
                    }

                    continue;
                }

                $row[$j] = str_replace(['\\', '\'', "\0", "\r\n"], ['\\\\', '\'\'', '\0', '\r\n'], $row[$j]);

                if (is_numeric($row[$j]) && stripos($tableColumns[$j], 'int') !== false) {
                    $data .= $row[$j];
                } else {
                    $data .= '\'' . $row[$j] . '\'';
                }

                if ($j < ($numFields - 1)) {
                    $data .= ', ';
                }
            }

            if ($countEntries < $count) {
                if ($countEntries % $this->maximumInsertLimit === 0) {
                    $data .= ");\n";
                    $data .= 'INSERT INTO ' . $this->db->quoteName($table) . ' VALUES' . "\n";
                } else {
                    $data .= "),\n";
                }
            }
        }

        $data .= ");\n";
        $this->writeDatabaseFile($data);
    }

    /**
     * Adds post process data
     *
     * @param string $hash
     *
     * @since 5.0.0.0-PRO
     */
    private function createBackupSqlArchiveDatabasePostProcess(string $hash): void
    {
        $postProcessData = $this->getPostProcessData($hash);

        if (empty($postProcessData)) {
            return;
        }

        $data = '';

        foreach ($postProcessData as $postProcessDatum) {
            if ($postProcessDatum['type'] === 'constraint') {
                $data .= 'ALTER TABLE ' . $this->db->quoteName($postProcessDatum['table']) . "\n" . '  ADD ' . $postProcessDatum['command'] . ";\n\n";
            }
        }

        $this->writeDatabaseFile($data);
    }

    /**
     * Creates a zip archive of the SQL dump file
     *
     * @return void
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    private function createBackupZipArchiveDatabaseAjax(): void
    {
        $zipDatabase = EasyJoomlaBackupHelper::openZipArchive($this->backupFolder . $this->fileName);

        if (!is_file($this->backupFolder . $this->fileNameDatabase)) {
            return;
        }

        $zipDatabase->addFile($this->backupFolder . $this->fileNameDatabase, $this->fileNameDatabase);
        $this->setExternalAttributes($zipDatabase, $this->fileNameDatabase, 33184);
        $this->encryptArchive($zipDatabase, $this->fileNameDatabase);

        EasyJoomlaBackupHelper::closeZipArchive($zipDatabase);

        File::delete($this->backupFolder . $this->fileNameDatabase);
    }

    /**
     * Deletes backup files from the server and the corresponding database entries
     *
     * @return bool
     * @throws Exception
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    public function delete(): bool
    {
        $ids = $this->input->get('id', 0, 'ARRAY');
        $table = new CreateTable($this->db);

        foreach ($ids as $id) {
            $table->load($id);

            $filePath = $this->backupFolder . $table->get('name');

            if (is_file($filePath)) {
                File::delete($filePath);
            }

            if (!$table->delete($id)) {
                throw new RuntimeException(Text::_('JERROR_AN_ERROR_HAS_OCCURRED'), 404);
            }
        }

        return true;
    }

    /**
     * Main function for the backup process - used in plugin and CLI script
     *
     * @param string $type
     * @param string $source
     *
     * @return bool
     * @throws Exception
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    public function createBackup(string $type, string $source = ''): bool
    {
        if (!class_exists('ZipArchive')) {
            return false;
        }

        $this->executionSource = $source;
        $this->externalAttributes = $this->checkExternalAttributes();
        $this->encryptArchive = EasyJoomlaBackupHelper::checkEncryptArchive();
        $this->encryptionMethod = $this->setEncryptionMethod();

        $start = microtime(true);
        $status = true;
        $statusDb = true;

        $this->fileName = $this->createFilename();

        if ($type === 'filebackup' || $type === 'fullbackup') {
            $status = $this->createBackupZipArchiveFiles();
        }

        if ($type === 'databasebackup' || $type === 'fullbackup') {
            $statusDb = $this->createBackupZipArchiveDatabase();
        }

        if (!empty($status) && !empty($statusDb)) {
            $table = new CreateTable($this->db);

            $data = [];
            $data['date'] = $this->backupDatetime->toSql();
            $data['type'] = $type;
            $data['name'] = $this->fileName;
            $data['size'] = filesize($this->backupFolder . $this->fileName);
            $data['duration'] = round(microtime(true) - $start, 2);
            $data['comment'] = $this->input->get('comment', '', 'STRING');

            if (!empty($source)) {
                Factory::getApplication()->getLanguage()->load('com_easyjoomlabackup', JPATH_ADMINISTRATOR);

                $data['comment'] = Text::_('COM_EASYJOOMLABACKUP_CRONJOBPLUGIN');

                if ($source === 'cli') {
                    $data['comment'] = Text::_('COM_EASYJOOMLABACKUP_CLISCRIPT');
                }
            }

            if (!$table->save($data)) {
                throw new RuntimeException(Text::_('JERROR_AN_ERROR_HAS_OCCURRED'), 404);
            }

            $data['dateLocal'] = $this->backupDatetime->toSql(true);
            asort($data);
            $this->dataStored = $data;

            return true;
        }

        return false;
    }

    /**
     * Creates a filename for the backup archive from the URL, the date and a random string
     *
     * @return string
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    private function createFilename(): string
    {
        $fileName = $this->getFileNamePrefix(true) . '_' . $this->backupDatetime->format('Y-m-d_H-i-s', true);

        if (is_file($this->backupFolder . $fileName)) {
            $fileName = $this->createFilename();
        }

        $addSuffix = $this->params->get('addSuffixArchive', 1);

        if (!empty($addSuffix)) {
            $fileName .= '_' . UserHelper::genRandomPassword(16);
        }

        $fileName .= '.zip';

        return $fileName;
    }

    /**
     * Creates the archive file of all files from the Joomla! installation with a possible
     * exclusion of files and folders
     *
     * @return bool
     * @version 5.0.1.0-PRO
     * @since   5.0.0.0-PRO
     */
    private function createBackupZipArchiveFiles(): bool
    {
        $this->excludeFiles = $this->getExcludeFiles();
        $this->excludeFolders = $this->getExcludeFolders();
        $this->excludeFoldersCreateEmpty = $this->getExcludeFoldersEmpty();

        if (!$dir = @opendir($this->pathRoot)) {
            return false;
        }

        $filesArray = [];

        while ($file = readdir($dir)) {
            if ($file === '.' || $file === '..') {
                continue;
            }

            if (is_dir($this->pathRoot . '/' . $file)) {
                if (strncmp($file, '.', 1) === 0) {
                    continue;
                }

                $zipFolder = EasyJoomlaBackupHelper::openZipArchive($this->backupFolder . $this->fileName);
                $this->zipFoldersAndFilesRecursive($zipFolder, $this->pathRoot . '/' . $file, $file, $file);
                EasyJoomlaBackupHelper::closeZipArchive($zipFolder);

                if ($zipFolder->status !== 0) {
                    return false;
                }
            } elseif (is_file($this->pathRoot . '/' . $file)) {
                $filesArray[] = $file;
            }
        }

        if (!empty($filesArray)) {
            $zipFile = EasyJoomlaBackupHelper::openZipArchive($this->backupFolder . $this->fileName);

            foreach ($filesArray as $file) {
                if ($this->isFileExcluded($file)) {
                    continue;
                }

                $zipFile->addFile($this->pathRoot . '/' . $file, $file);
                $this->setExternalAttributes($zipFile, $file, fileperms($this->pathRoot . '/' . $file));
                $this->encryptArchive($zipFile, $file);
            }

            EasyJoomlaBackupHelper::closeZipArchive($zipFile);

            if ($zipFile->status !== 0) {
                return false;
            }
        }

        closedir($dir);
        unset($zipFolder, $zipFile);

        return true;
    }

    /**
     * Loads all files and (sub-)folders for the zip archive recursively
     *
     * @param object $zip
     * @param string $folder
     * @param string $folderRelative
     * @param string $folderStart
     *
     * @return void
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    private function zipFoldersAndFilesRecursive(object $zip, string $folder, string $folderRelative, string $folderStart = ''): void
    {
        if (!empty($folderStart)) {
            if (in_array($folderStart, $this->excludeFoldersCreateEmpty, true)) {
                $zip->addEmptyDir($folderStart . '/');
                $this->setExternalAttributes($zip, $folderStart . '/', 16877);

                $zip->addFromString($folderStart . '/index.html', '');
                $this->setExternalAttributes($zip, $folderStart . '/index.html', 33188);
                $this->encryptArchive($zip, $folderStart . '/index.html');

                return;
            }

            if (!empty($this->excludeFolders) && in_array($folderStart, $this->excludeFolders, true)) {
                return;
            }

            $zip->addEmptyDir($folderStart . '/');
            $this->setExternalAttributes($zip, $folderStart . '/', fileperms($folder . '/'));
        }

        if (!$dir = @opendir($folder)) {
            return;
        }

        while ($file = readdir($dir)) {
            if ($file !== '.' && $file !== '..' && is_dir($folder . '/' . $file)) {
                if (in_array($folderRelative . '/' . $file, $this->excludeFoldersCreateEmpty, true)) {
                    $zip->addEmptyDir($folderRelative . '/' . $file . '/');
                    $this->setExternalAttributes($zip, $folderRelative . '/' . $file . '/', 16877);

                    $zip->addFromString($folderRelative . '/' . $file . '/index.html', '');
                    $this->setExternalAttributes($zip, $folderRelative . '/' . $file . '/index.html', 33188);
                    $this->encryptArchive($zip, $folderRelative . '/' . $file . '/index.html');

                    if ($folderRelative . '/' . $file === 'administrator/components/com_easyjoomlabackup/backups' || $folderRelative . '/' . $file === EasyJoomlaBackupHelper::getBackupStorageLocation(false, true)) {
                        $zip->addFromString($folderRelative . '/' . $file . '/.htaccess', 'Deny from all');
                        $this->setExternalAttributes($zip, $folderRelative . '/' . $file . '/.htaccess', 33060);
                        $this->encryptArchive($zip, $folderRelative . '/' . $file . '/.htaccess');
                    }

                    continue;
                }

                if (!empty($this->excludeFolders) && in_array($folderRelative . '/' . $file, $this->excludeFolders, true)) {
                    continue;
                }

                $zip->addEmptyDir($folderRelative . '/' . $file . '/');
                $this->setExternalAttributes($zip, $folderRelative . '/' . $file . '/', fileperms($folder . '/' . $file . '/'));

                $this->zipFoldersAndFilesRecursive($zip, $folder . '/' . $file, $folderRelative . '/' . $file);
            } elseif (is_file($folder . '/' . $file)) {
                if ($this->isFileExcluded($folderRelative . '/' . $file)) {
                    continue;
                }

                $zip->addFile($folder . '/' . $file, $folderRelative . '/' . $file);
                $this->setExternalAttributes($zip, $folderRelative . '/' . $file, fileperms($folder . '/' . $file));
                $this->encryptArchive($zip, $folderRelative . '/' . $file);
            }
        }

        closedir($dir);
    }

    /**
     * Creates a complete dump of the Joomla! database
     *
     * @return bool
     * @since 5.0.0.0-PRO
     */
    private function createBackupZipArchiveDatabase(): bool
    {
        $zipDatabase = EasyJoomlaBackupHelper::openZipArchive($this->backupFolder . $this->fileName);

        $hash = md5($this->fileNameDatabase);
        $this->setFileNameDatabase();
        $this->backupDatabase($hash);

        $zipDatabase->addFile($this->backupFolder . $this->fileNameDatabase, $this->fileNameDatabase);
        $this->setExternalAttributes($zipDatabase, $this->fileNameDatabase, 33184);
        $this->encryptArchive($zipDatabase, $this->fileNameDatabase);

        EasyJoomlaBackupHelper::closeZipArchive($zipDatabase);

        unlink($this->backupFolder . $this->fileNameDatabase);

        return !($zipDatabase->status !== 0);
    }

    /**
     * Creates a SQL Dump of the Joomla! database and add it directly to the archive
     *
     * @param string $hash
     *
     * @return void
     * @since 5.0.0.0-PRO
     */
    private function backupDatabase(string $hash): void
    {
        $tables = $this->getDatabaseTables();
        $data = EasyJoomlaBackupHelper::getDatabaseDumpHeader();
        $this->writeDatabaseFile($data);

        foreach ($tables as $table) {
            if (!$this->isDbTableValid($table)) {
                continue;
            }

            if ($this->addDropStatement) {
                $data = 'DROP TABLE IF EXISTS ' . $this->db->quoteName($table) . ';' . "\n";
                $this->writeDatabaseFile($data);
            }

            $this->db->setQuery('SHOW CREATE TABLE ' . $table);
            $rowCreate = $this->db->loadRow();
            $this->checkPostProcessDataCreateTable($rowCreate, $hash);

            $data = $rowCreate[1] . ";\n\n";
            $this->writeDatabaseFile($data);

            $tableInformation = $this->db->getTableColumns($table);
            $numFields = count($tableInformation);
            $tableColumnTypes = array_values($tableInformation);

            $this->db->setQuery('SELECT * FROM ' . $this->db->quoteName($table));
            $this->db->execute();
            $count = $this->db->getNumRows();

            if ($count > 0) {
                $passes = (int)ceil($count / $this->maximumListLimit);
                $this->iterator = $this->db->getIterator();
                $this->db->setQuery('SELECT * FROM ' . $this->db->quoteName($table));

                for ($round = 0; $round < $passes; $round++) {
                    $rowList = $this->getRowList($passes);
                    $this->addRowEntries($rowList, $numFields, $tableColumnTypes, $table);
                }
            }

            $data .= "\n\n";
            $this->writeDatabaseFile($data);
        }

        $this->createBackupSqlArchiveDatabasePostProcess($hash);
    }

    /**
     * Loads the correct backup archive and creates the download process
     *
     * @return bool|void
     * @throws Exception
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    public function download()
    {
        $id = $this->input->get('id', 0, 'INTEGER');
        $table = new CreateTable($this->db);

        $table->load($id);
        $file = $this->backupFolder . $table->get('name');

        if (!file_exists($file)) {
            return false;
        }

        header('Pragma: public');
        header('Expires: 0');
        header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
        header('Cache-Control: public');
        header('Content-Description: File Transfer');
        header('Content-Type: application/zip', true, 200);
        header('Content-Disposition: attachment; filename=' . $table->get('name'));
        header('Content-Transfer-Encoding: binary');
        header('Content-Length: ' . $table->get('size'));

        $chunkSize = 10 * (1024 * 1024);
        $handle = fopen($file, 'rb');

        while (!feof($handle)) {
            $buffer = fread($handle, $chunkSize);
            echo $buffer;
            ob_flush();
            flush();
        }

        fclose($handle);

        exit();
    }

    /**
     * Uploads the archive automatically to Dropbox
     *
     * @param string $type
     * @param string $fileName
     *
     * @return bool
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public function dropboxUploadAutomatic(string $type = 'ui', string $fileName = ''): bool
    {
        if ($fileName === '') {
            $fileName = $this->fileName;
        }

        if ($type === 'ui') {
            $dropboxUploadAutomatic = (bool)$this->params->get('dropboxUploadAutomatic', false);

            if ($dropboxUploadAutomatic) {
                $this->dropboxUpload($fileName);
                EasyJoomlaBackupHelper::resetDropboxUploadStatus();

                return true;
            }

            return false;
        }

        if ($type === 'cron' || $type === 'plugin') {
            $dropboxUploadAutomaticCron = (bool)$this->params->get('dropboxUploadAutomaticCron', false);

            if ($dropboxUploadAutomaticCron) {
                $this->dropboxUpload($fileName);
                EasyJoomlaBackupHelper::resetDropboxUploadStatus();

                return true;
            }
        }

        return false;
    }

    /**
     * Uploads the selected archive to Dropbox
     *
     * @param string $fileName
     *
     * @return bool
     * @throws Exception
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    public function dropboxUpload(string $fileName = ''): bool
    {
        if ($fileName === '') {
            $id = $this->input->get('id', 0, 'INTEGER');
            $table = new CreateTable($this->db);

            $table->load($id);
            $fileName = $table->get('name');
        }

        $file = $this->backupFolder . $fileName;

        if (!is_file($file)) {
            return false;
        }

        if (!EasyJoomlaBackupHelper::isUploadAllowed('dropbox')) {
            return false;
        }

        $dropboxUploadAccessToken = $this->params->get('dropboxUploadAccessToken', '');
        $dropbox = new DropboxHelper($dropboxUploadAccessToken);
        $result = $dropbox->upload('/' . $fileName, $file);

        if ($result) {
            return true;
        }

        return false;
    }

    /**
     * Uploads the selected archive to a SFTP server
     *
     * @param string $fileName
     *
     * @return bool
     * @throws Exception
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    public function sftpUpload(string $fileName = ''): bool
    {
        if ($fileName === '') {
            $id = $this->input->get('id', 0, 'INTEGER');
            $table = new CreateTable($this->db);

            $table->load($id);
            $fileName = $table->get('name');
        }

        $file = $this->backupFolder . $fileName;

        if (!is_file($file)) {
            return false;
        }

        if (!EasyJoomlaBackupHelper::isUploadAllowed('sftp')) {
            return false;
        }

        $sftp = EasyJoomlaBackupHelper::getSftp();

        if ($sftp === null) {
            return false;
        }

        $result = $sftp->put($fileName, $file, SFTP::SOURCE_LOCAL_FILE);

        if ($result) {
            return true;
        }

        return false;
    }

    /**
     * Sends a notification mail after the backup process
     *
     * @param string $type
     * @param bool   $status
     * @param string $outputType
     *
     * @return bool
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public function sendNotification(string $type, bool $status, string $outputType = 'default'): bool
    {
        if ($type !== 'cron' && $type !== 'plugin') {
            return false;
        }

        $notificationMailCron = (bool)$this->params->get('notificationMailCron', false);

        if (!$notificationMailCron) {
            return false;
        }

        $notificationMailError = (bool)$this->params->get('notificationMailError', false);

        if ($notificationMailError && $status) {
            return false;
        }

        $notificationMailCronAddresses = array_map('trim', explode(',', $this->params->get('notificationMailCronAddresses')));
        $urlRoot = $this->params->get('urlRoot', Uri::root());

        if (empty($notificationMailCronAddresses)) {
            return false;
        }

        EasyJoomlaBackupHelper::sendNotificationMail($notificationMailCronAddresses, $status, $urlRoot, $this->dataStored, $outputType);

        return true;
    }

    /**
     * Checks whether more backup files are available than allowed and starts deletion process if required
     *
     * @return bool
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    public function removeBackupFilesMax(): bool
    {
        $maxNumberBackups = (int)$this->params->get('maxNumberBackups', 5);

        if ($maxNumberBackups === 0) {
            return false;
        }

        $totalNumberBackups = $this->getTotal();

        if ($totalNumberBackups <= $maxNumberBackups) {
            return false;
        }

        if ($this->deleteFilesMax($maxNumberBackups, $totalNumberBackups - $maxNumberBackups)) {
            return true;
        }

        return false;
    }

    /**
     * Gets the total number of entries after the backup process was executed
     *
     * @return int
     * @since 5.0.0.0-PRO
     */
    private function getTotal(): int
    {
        $query = $this->db->getQuery(true);
        $query->select('*');
        $query->from('#__easyjoomlabackup');

        return $this->_getListCount($query);
    }

    /**
     * Deletes all unneeded files. The number of files which should be kept can be set in the settings
     *
     * @param int $limitstart
     * @param int $limit
     *
     * @return bool
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    private function deleteFilesMax(int $limitstart, int $limit): bool
    {
        $query = $this->db->getQuery(true);
        $query->select($this->db->quoteName('id'));
        $query->from('#__easyjoomlabackup');
        $query->order($this->db->escape('date DESC'));

        $data = $this->_getList($query, $limitstart, $limit);

        if (empty($data)) {
            return false;
        }

        $ids = [];

        foreach ($data as $value) {
            $ids[] = $value->id;
        }

        $this->input->set('id', $ids);

        if (!$this->delete()) {
            return false;
        }

        return true;
    }

    /**
     * Gets the stored data created after a successful backup creation
     *
     * @return array
     * @since 5.0.0.0-PRO
     */
    public function getStoredData(): array
    {
        return $this->dataStored;
    }
}
