<?php
/**
 * @copyright	Copyright (C) 2011 Simplify Your Web, Inc. All rights reserved.
 * @license		GNU General Public License version 3 or later; see LICENSE.txt
 */

namespace SYW\Component\WeblinkLogosPro\Site\Helper;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Filesystem\File;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Log\Log;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\CMS\Uri\Uri;
use Joomla\Registry\Registry;
use SYW\Library\Image as SYWImage;

/**
 * Class Helper
 */
class Helper
{
	protected static $weblinks_config_params;

	protected static $image_extension_types = array('png', 'jpg', 'jpeg', 'gif', 'webp', 'avif');

	/**
	 * Get an instance of the named model
	 *
	 * @param   string  $name  Model name
	 * @return null|object
	 */
	public static function getModel($name)
	{
		$model = null;

		if (file_exists(JPATH_SITE . '/components/com_weblinklogospro/models/' . strtolower($name) . '.php')) {
			require_once JPATH_SITE . '/components/com_weblinklogospro/models/' . strtolower($name) . '.php';
			$model = BaseDatabaseModel::getInstance($name, 'WeblinklogosproModel');
		}

		return $model;
	}

	/**
	 * Load the script that handles click feedback
	 */
	public static function loadClickedScript() {

		$wam = Factory::getApplication()->getDocument()->getWebAssetManager();

		$script = 'document.addEventListener("readystatechange", function(event) { ';
			$script .= 'if (event.target.readyState === "complete") { ';
				$script .= 'var items = document.querySelectorAll(".wlp_directory .weblink_item"); ';
				$script .= 'for (var i = 0; i < items.length; i++) { ';
					$script .= 'var links = items[i].querySelectorAll("a"); ';
					$script .= 'for (var j = 0; j < links.length; j++) { ';
						$script .= 'links[j].addEventListener("click", function(e) { ';
							$script .= 'this.classList.add("clicked");';
						$script .= '}.bind(items[i])); '; // the binding here replaces 'this' with items[i]
					$script .= '} ';
				$script .= '} ';
			$script .= '} ';
		$script .= '}); ';

// 		$script = 'jQuery(document).ready(function($) { ';
// 			$script .= '$(".wlp_directory .weblink_item a").click(function() { ';
// 				$script .= '$(this).closest("li").addClass("clicked"); ';
// 			$script .= '}); ';
// 		$script .= '});';

		$wam->addInlineScript($script);
	}

	/**
	* Create the thumbnail(s), if possible
	*
	* @param string $suffix
	* @param string $item_id
	* @param string $imagesrc
	* @param string $tmp_path
	* @param boolean $clear_cache
	* @param integer $head_width
	* @param integer $head_height
	* @param boolean $crop_picture
	* @param array $image_quality_array
	* @param string $filter
	* @param boolean $create_high_resolution
	*
	* @return array the original image path if errors before thumbnail creation
	*  or no thumbnail path if errors during thumbnail creation
	*  or thumbnail path if no error
	*/
	public static function getThumbnailPath($suffix, $item_id, $imagesrc, $tmp_path, $clear_cache, $head_width, $head_height, $crop_picture, $image_quality_array, $filter, $create_highres_images = false, $allow_remote = true, $thumbnail_mime_type = '')
	{
		$result = array(null, null); // image link and error

		if ($head_width == 0 || $head_height == 0) {
			// keep original image
			$result[0] = $imagesrc;
			$result[1] = Text::_('COM_WEBLINKLOGOSPRO_INFO_USINGORIGINALIMAGE'); // necessary to avoid high resolution syntax

			return $result;
		}

		if (!extension_loaded('gd') && !extension_loaded('imagick')) {
			// missing gd library
			$result[0] = $imagesrc;
			$result[1] = Text::_('COM_WEBLINKLOGOSPRO_ERROR_GDNOTLOADED');

			return $result;
		}

		$original_imagesrc = $imagesrc;

		// there may be extra info in the path
		// example: http://www.tada.com/image.jpg?x=3
		// thubmnails cannot be created if ? in the path

		$url_array = explode("?", $imagesrc);
		$imagesrc = $url_array[0];

		$imageext = strtolower(File::getExt($imagesrc));
		$original_imageext = $imageext;

		if (!in_array($imageext, self::$image_extension_types)) {

			// case where image is a URL with no extension (generated image)
			// example: http://argos.scene7.com/is/image/Argos/7491801_R_Z001A_UC1266013?$TMB$&wid=312&hei=312
			// thubmnails cannot be created from generated images external paths
			// or image has another file type like .tiff

			$result[0] = $original_imagesrc;
			$result[1] = Text::sprintf('COM_WEBLINKLOGOSPRO_ERROR_UNSUPPORTEDFILETYPE', $original_imagesrc);

			return $result;
		}

		// URL works only if 'allow url fopen' is 'on', which is a security concern
		// retricts images to the ones found on the site, external URLs are not allowed (for security purposes)
		if (substr_count($imagesrc, 'http') <= 0) { // if the image is internal
			if (substr($imagesrc, 0, 1) == '/') {
				// take the slash off
				$imagesrc = ltrim($imagesrc, '/');
			}
		} else {
			$base = Uri::base(); // Uri::base() is http://www.mysite.com/subpath/
			$imagesrc = str_ireplace($base, '', $imagesrc);
		}

		// we end up with all $imagesrc paths as 'images/...'
		// if not, the URL was from an external site

		if (substr_count($imagesrc, 'http') > 0) {
			// we have an external URL
		    if (/*!ini_get('allow_url_fopen') || */!$allow_remote) {
				$result[0] = $original_imagesrc;
				$result[1] = Text::sprintf('COM_WEBLINKLOGOSPRO_ERROR_EXTERNALURLNOTALLOWED', $imagesrc);

				return $result;
			}
		}

		switch ($thumbnail_mime_type) {
		    case 'image/jpg': $imageext = 'jpg'; break;
		    case 'image/png': $imageext = 'png'; break;
		    case 'image/webp': $imageext = 'webp'; break;
		    case 'image/avif': $imageext = 'avif';
		}

		if ($filter == 'none' || strpos($filter, '_css') !== false) {
			$filtername = '';
		} else {
			$filtername = '_'.$filter;
		}

		if (!empty($suffix)) {
			$suffix = '_'.$suffix;
		}

		$filename = $tmp_path.'/thumb'.$suffix.'_'.$item_id.$filtername.'.'.$imageext;
		$filename_highres = $tmp_path.'/thumb'.$suffix.'_'.$item_id.$filtername.'@2x.'.$imageext;
		if ((is_file(JPATH_ROOT.'/'.$filename) && !$clear_cache && !$create_highres_images)
			|| (is_file(JPATH_ROOT.'/'.$filename) && !$clear_cache && $create_highres_images && is_file(JPATH_ROOT.'/'.$filename_highres))) {

			// thumbnail(s) already exist

		} else { // create the thumbnail

			$image = new SYWImage($imagesrc);

			if (is_null($image->getImagePath())) {
				$result[1] = Text::sprintf('COM_WEBLINKLOGOSPRO_ERROR_IMAGEFILEDOESNOTEXIST', $imagesrc);
			} else if (is_null($image->getImageMimeType())) {
				$result[1] = Text::sprintf('COM_WEBLINKLOGOSPRO_ERROR_UNABLETOGETIMAGEPROPERTIES', $imagesrc);
			} else if (is_null($image->getImage()) || $image->getImageWidth() == 0) {
				$result[1] = Text::sprintf('COM_WEBLINKLOGOSPRO_ERROR_UNSUPPORTEDFILETYPE', $imagesrc);
			} else {

			    $quality = self::getImageQualityFromExt($imageext, $image_quality_array);

				// negative values force the creation of the thumbnails with size of original image
				// great to create high-res of original image and/or to use quality parameters to create an image with smaller file size
				if ($head_width < 0 || $head_height < 0) {
					$head_width = $image->getImageWidth();
					$head_height = $image->getImageHeight();
				}

				if ($image->toThumbnail($filename, $thumbnail_mime_type, $head_width, $head_height, $crop_picture, $quality, $filter, $create_highres_images)) {

				    if ($image->getImageMimeType() === 'image/webp' || $thumbnail_mime_type === 'image/webp' || $image->getImageMimeType() === 'image/avif' || $thumbnail_mime_type === 'image/avif') { // create fallback

				        $fallback_extension = 'png';
				        $fallback_mime_type = 'image/png';

				        // create fallback with original image mime type when the original is not webp or avif
				        if ($image->getImageMimeType() !== 'image/webp' && $image->getImageMimeType() !== 'image/avif') {
				            $fallback_extension = $original_imageext;
				            $fallback_mime_type = $image->getImageMimeType();
				        }

				        $quality = self::getImageQualityFromExt($fallback_extension, $image_quality_array);

				        if (!$image->toThumbnail($tmp_path . '/thumb' . $suffix . '_' . $item_id . $filtername . '.' . $fallback_extension, $fallback_mime_type, $head_width, $head_height, $crop_picture, $quality, $filter, $create_highres_images)) {
				            $result[1] = Text::sprintf('COM_WEBLINKLOGOSPRO_ERROR_THUMBNAILCREATIONFAILED', $imagesrc);
				        }
					}
				} else {
				    $result[1] = Text::sprintf('COM_WEBLINKLOGOSPRO_ERROR_THUMBNAILCREATIONFAILED', $imagesrc);
				}
			}

			$image->destroy();
		}

		if (empty($result[1])) {
			$result[0] = $filename;
		}

		return $result;
	}

	static protected function getImageQualityFromExt($image_extension, $qualities = array('jpg' => 75, 'png' => 3, 'webp' => 80, 'avif' => 80))
	{
	    $quality = -1;

	    switch ($image_extension){
	        case 'jpg': case 'jpeg': $quality = $qualities['jpg']; break; // 0 to 100
	        case 'png': $quality = round(11.111111 * (9 - $qualities['png'])); break; // compression: 0 to 9
	        case 'webp': $quality = $qualities['webp']; break; // 0 to 100
	        case 'avif': $quality = $qualities['avif']; // 0 to 100
	    }

	    return $quality;
	}

	/**
	 * Delete all thumbnails for a view
	 *
	 * @param string $tmp_path
	 * @param string $view
	 *
	 * @return false if the glob function failed, true otherwise
	 */
	static function clearThumbnails($tmp_path, $view = '') {

		Log::addLogger(array('text_file' => 'syw.errors.php'), Log::ALL, array('syw'));

		if ($view) {
			$view = '_'.$view;
		}

		if (function_exists('glob')) {
			$filenames = glob(JPATH_ROOT.'/'.$tmp_path.'/thumb'.$view.'_*.*');
			if ($filenames == false) {
				Log::add('WeblinklogosproFrontendHelper:clearThumbnails() - Error on glob - No permission on files/folder or old system', Log::ERROR, 'syw');
				return false;
			}

			foreach ($filenames as $filename) {
				File::delete($filename); // returns false if deleting failed - won't log to avoid making the log file huge
			}

			return true;
		} else {
			Log::add('WeblinklogosproFrontendHelper:clearThumbnails() - glob - function does not exist', Log::ERROR, 'syw');
		}

		return false;
	}

	/**
	 * Load user stylesheet for a view
	 */
	static function loadUserStylesheet($view)
	{
		$wam = Factory::getApplication()->getDocument()->getWebAssetManager();

		$prefix = 'common_user';

		if (File::exists(JPATH_ROOT . '/media/com_weblinklogospro/css/' . $view . '/' . $prefix . '_styles-min.css')) {
		    if (JDEBUG && File::exists(JPATH_ROOT . '/media/com_weblinklogospro/css/' . $view . '/' . $prefix.'_styles.css')) {
		        $wam->registerAndUseStyle('wlp.' . $view . '_' . $prefix . '_styles', 'com_weblinklogospro/' . $view . '/' . $prefix . '_styles.css', ['relative' => true, 'version' => 'auto']);
		    } else {
		        $wam->registerAndUseStyle('wlp.' . $view . '_' . $prefix . '_styles', 'com_weblinklogospro/' . $view . '/' . $prefix . '_styles-min.css', ['relative' => true, 'version' => 'auto']);
		    }
		} else {
		    $wam->registerAndUseStyle('wlp.' . $view . '_' . $prefix . '_styles', 'com_weblinklogospro/' . $view . '/' . $prefix . '_styles.min.css', ['relative' => true, 'version' => 'auto']);
		}
	}

	static function getLimitsForBox($specific_list = '')
	{
		$limits = array();

		if ($specific_list) {
			$numbers = explode(',', $specific_list);
			foreach ($numbers as $number) {
				$number = trim($number);
				if ($number != '') { // otherwise will remove 0s
					if ($number !== '0') {
						$limits[] = HTMLHelper::_('select.option', $number);
					} else {
						$limits[] = HTMLHelper::_('select.option', '0', Text::_('JALL'));
					}
				}
			}

		} else {
		    for ($i = 5; $i <= 30; $i += 5) {
		    	$limits[] = HTMLHelper::_('select.option', "$i");
		    }

		    $limits[] = HTMLHelper::_('select.option', '50', Text::_('J50'));
		    $limits[] = HTMLHelper::_('select.option', '100', Text::_('J100'));
		    $limits[] = HTMLHelper::_('select.option', '0', Text::_('JALL'));
		}

	    return $limits;
	}

	/**
	 * Get the site mode
	 * @return string (dev|prod|adv)
	 */
	public static function getSiteMode($params)
	{
		return $params->get('site_mode', 'adv');
	}

	/**
	 * Is the picture cache set to be cleared
	 * @return boolean
	 */
	public static function IsClearPictureCache($params)
	{
		if (self::getSiteMode($params) == 'dev') {
			return true;
		}
		if (self::getSiteMode($params) == 'prod') {
			return false;
		}
		return boolval($params->get('clear_cache', true));
	}

	/**
	 * Is the style/script cache set to be cleared
	 * @return boolean
	 */
	public static function IsClearHeaderCache($params)
	{
		if (self::getSiteMode($params) == 'dev') {
			return true;
		}
		if (self::getSiteMode($params) == 'prod') {
			return false;
		}
		return boolval($params->get('clear_header_files_cache', 'true'));
	}

	/**
	 * Are errors shown ?
	 * @return boolean
	 */
	public static function isShowErrors($params)
	{
		if (self::getSiteMode($params) == 'dev') {
			return true;
		}
		if (self::getSiteMode($params) == 'prod') {
			return false;
		}
		return boolval($params->get('show_errors', false));
	}

	/**
	 * Are white spaces removed ?
	 * @return boolean
	 */
	public static function isRemoveWhitespaces($params)
	{
		if (self::getSiteMode($params) == 'dev') {
			return false;
		}
		if (self::getSiteMode($params) == 'prod') {
			return true;
		}
		return boolval($params->get('remove_whitespaces', false));
	}

	/**
	 * Get the Weblinks component's configuration parameters
	 * @return \Joomla\Registry\Registry
	 */
	public static function getWeblinksConfig()
	{
		if (!isset(self::$weblinks_config_params)) {

			self::$weblinks_config_params = new Registry();

			if (File::exists(JPATH_ADMINISTRATOR . '/components/com_weblinks/config.xml')) {
				self::$weblinks_config_params = ComponentHelper::getParams('com_weblinks');
			}
		}

		return self::$weblinks_config_params;
	}

}
