<?php

/**
 * @copyright
 * @package    Easy Joomla Backup Pro - EJB for Joomla! 5.x
 * @author     Viktor Vogel <admin@kubik-rubik.de>
 * @version    5.0.1.0-PRO - 2024-01-29
 * @link       https://kubik-rubik.de/ejb-easy-joomla-backup
 *
 * @license    GNU/GPL
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

namespace KubikRubik\Plugin\System\EasyJoomlaBackupCli\Console;

defined('_JEXEC') || die();

use Exception;
use Joomla\Console\Command\AbstractCommand;
use KubikRubik\Component\EasyJoomlaBackup\Administrator\Model\CreateModel;
use Symfony\Component\Console\{Command\Command, Input\InputDefinition, Input\InputInterface, Input\InputOption, Output\OutputInterface, Style\SymfonyStyle};
use KubikRubik\Component\EasyJoomlaBackup\Administrator\Helper\EasyJoomlaBackupHelper;

use function defined;

/**
 * Console command for purging the expired cache
 *
 * @version 5.0.0.1-PRO
 * @since   5.0.0.0-PRO
 */
class DatabaseCommand extends AbstractCommand
{
    /**
     * @var string $defaultName
     * @since 5.0.0.0-PRO
     */
    protected static $defaultName = 'backup:database';

    /**
     * @var bool $json
     * @since 5.0.0.0-PRO
     */
    private bool $json;

    /**
     * @var bool $raw
     * @since 5.0.0.0-PRO
     */
    private bool $raw;

    /**
     * @var SymfonyStyle $symfonyStyle
     * @since 5.0.0.0-PRO
     */
    private SymfonyStyle $symfonyStyle;

    /**
     * Internal function to execute the command
     *
     * @param InputInterface  $input
     * @param OutputInterface $output
     *
     * @return int
     * @throws Exception
     * @version 5.0.0.1-PRO
     * @since   5.0.0.0-PRO
     */
    protected function doExecute(InputInterface $input, OutputInterface $output): int
    {
        $this->symfonyStyle = new SymfonyStyle($input, $output);
        $this->json = (bool)$input->getOption('json');
        $this->raw = (bool)$input->getOption('raw');

        $this->writeConsoleOutput('Easy Joomla Backup - Database Backup', 'title');

        EasyJoomlaBackupHelper::increaseServerLimits();
        $_SERVER['HTTP_HOST'] = '';
        $_SERVER['REQUEST_URI'] = '';

        $model = new CreateModel();
        $this->writeConsoleOutput('Backup process is being executed. Please be patient!', 'text');

        if ($model->createBackup('databasebackup', 'cli')) {
            $model->dropboxUploadAutomatic('cron');
            $model->sendNotification('cron', true, $this->json ? 'json' : 'default');
            $model->removeBackupFilesMax();
            $this->writeConsoleOutput('Database backup created successfully', 'success');

            $storedData = $model->getStoredData();
            $storedData['status'] = true;

            return $this->writeConsoleOutputClose(Command::SUCCESS, $storedData);
        }

        $model->sendNotification('cron', false, $this->json ? 'json' : 'default');
        $this->writeConsoleOutput('Database backup could not be created', 'error');

        $storedData['status'] = false;

        return $this->writeConsoleOutputClose(Command::FAILURE, $storedData);
    }

    /**
     * Writes the console output
     *
     * @param string $message
     * @param string $type
     *
     * @since 5.0.0.0-PRO
     */
    private function writeConsoleOutput(string $message, string $type): void
    {
        if ($this->json || $this->raw) {
            return;
        }

        $this->symfonyStyle->$type($message);
    }

    /**
     * Write the closing console output
     *
     * @param int   $status
     * @param array $storedData
     *
     * @return int
     * @throws Exception
     * @since 5.0.0.0-PRO
     */
    private function writeConsoleOutputClose(int $status, array $storedData = []): int
    {
        if ($this->raw) {
            $this->symfonyStyle->write($status);
        }

        if ($this->json) {
            $this->symfonyStyle->write(json_encode($storedData, JSON_THROW_ON_ERROR));
        }

        return $status;
    }

    /**
     * Configures the command
     *
     * @since 5.0.0.0-PRO
     */
    protected function configure(): void
    {
        $help = '<info>%command.name%</info> will execute the database backup' . "\n";
        $help .= 'Usage: <info>php %command.full_name%</info>';

        $this->setDescription('Executes the database backup');
        $this->setHelp($help);

        $this->setDefinition(
            new InputDefinition(
                [
                    new InputOption('json', null, InputOption::VALUE_NONE),
                    new InputOption('raw', null, InputOption::VALUE_NONE),
                ]
            )
        );
    }
}
